# Copyright (c) 2011, 2013, Oracle and/or its affiliates. All rights reserved. 
#

=head1 NAME

  crspatch.pm  Oracle clusterware Patching Module/Package

=head1 DESCRIPTION

   This package contains functions required for  patching
   Oracle clusterware Software

=cut

#    MODIFIED   (MM/DD/YY)
#    shullur     12/25/12 - For moving the perform_CHM_upgrade before calling
#                           the s_validate_olrconfig
#    xyuan       12/23/12 - Fix bug 16032732
#    xyuan       12/20/12 - set env var ORAASM_UPGRADE in sub
#                           getActivePatchLevel & getSoftwarePatch
#    xyuan       11/22/12 - Fix bug 14845507
#    ssprasad    11/13/12 - Disable OKA actions for 12.1
#    ssprasad    10/22/12 - Add OKA actions for patching.
#    xyuan       10/16/12 - Fix checkpoint issues introduced when adding afd
#                           patching
#    xyuan       10/16/12 - Fix bug 14747053
#    xyuan       09/03/12 - Fix bug 14568610 - reset postatch checkpoints for
#                           different patch scenarios
#    shmubeen    07/31/12 - add afd patching
#    xyuan       08/02/12 - Fix a typo in the trace message
#    xyuan       07/31/12 - Fix bug 14374241
#    xyuan       07/24/12 - Fix bug 14313040
#    rtamezd     06/19/12 - Use print_lines()
#    xyuan       06/07/12 - Disable/enable CRS resources only for '-norestart'
#    rtamezd     05/16/12 - Fix bug 14068782
#    anjiwaji    05/08/12 - Rename
#                           createACFSDriversResource->actionDriversResource
#                           and pass in an action
#    sidshank    05/03/12 - remove s_redirect/restore output subroutines
#    xyuan       03/25/12 - Fix bug 13867795
#    xyuan       03/25/12 - Fix bug 13872932 
#    xyuan       03/25/12 - Fix bug 13879808
#    xyuan       02/27/12 - Remove unnecessary subroutines
#    xyuan       02/20/12 - New patching interfaces for 12c
#    rtamezd     01/11/12 - Fix bug 13576465
#    anjiwaji    12/14/11 - Create and start driver resource after patching.
#    shullur     11/03/11 - XbranchMerge shullur_bug_11852891 from
#                           st_has_11.2.0
#    xesquive    08/15/11 - forward merge from bug-12587677
#    xyuan       07/27/11 - XbranchMerge xyuan_bug-12701521 from
#                           st_has_11.2.0.3.0
#    ksviswan    03/08/11 - opatch auto segregation
#    ksviswan    09/23/10 - Part fix for bug 10119895
#    ksviswan    09/21/10 - Merge fix for bugs 9482228,9750739
#    dpham       06/30/10 - Add arguement to create_dirs() and set_file_perms() 
#                           functions (9850696)
#    ksviswan    08/24/09 - Fix Bug 8797450
#    dpham       07/29/09 - XbranchMerge dpham_bug-8727340 from
#                           st_has_11.2.0.1.0
#    ksviswan    07/24/09 - Install ACFS after patching
#    dpham       07/15/09 - XbranchMerge dpham_bug-8664938 from main
#    dpham       07/09/09 - wait for crs to start
#    ksviswan    04/20/09 - Creation

package crspatch;

use strict;
use English;
use File::Spec::Functions;
use Term::ANSIColor;

use constant CRSPATCH_SUCCESS     => 1;
use constant CRSPATCH_FAIL        => 0;

# root scripts modules
use crsutils;
use s_crsutils;
use oraacfs;
use crska;
use crsupgrade;

# export functions
use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);

@ISA = qw(Exporter);

my @exp_const = qw(CRSPATCH_SUCCESS CRSPATCH_FAIL);

my @exp_func  = qw(unlockPatchDestHome);

push @EXPORT, @exp_const, @exp_func;

sub new {
   shift;
   crsutils->new(@_);

   # patch
   if ($CFG->PREPATCH)
   {
     if ($CFG->SIHA) {
       unlockHAHomeforpatch();
     }
     else
     {
       crsPrePatch();
     }
   }
   elsif ($CFG->POSTPATCH)
   {
     if ($CFG->SIHA) {
       HAPatch();
     }
     else {
       crsPostPatch();
     }
   }
   elsif ($CFG->HAPatch)
   {
     HAPatch();
   }
   else #rootcrs.pl -patch
   {
     CRSPatch();
   }

}


sub Instantiatepatchfiles
{
   #TODO - Should we just rely on crsconfig_params or
   #should we derive the critical values.
   instantiate_scripts ();

   add_localOlr_OlrConfig_OcrConfig();

   my @crsconfig_dirs = read_file (catfile($CFG->ORA_CRS_HOME,
                                           'crs', 'utl', 'crsconfig_dirs'));
   create_dirs (\@crsconfig_dirs);

   copy_wrapper_scripts ();

   my @crsconfig_fileperms = read_file (catfile($CFG->ORA_CRS_HOME,
                                                'crs', 'utl', 'crsconfig_fileperms'));
   set_file_perms (\@crsconfig_fileperms);

   #set the ownership/permission of olr
   if ($CFG->SIHA) {
     s_set_ownergroup ($CFG->params('ORACLE_OWNER'),
                     $CFG->params('ORA_DBA_GROUP'), $CFG->OLR_LOCATION);
   } else {
     s_set_ownergroup ($CFG->SUPERUSER,
                     $CFG->params('ORA_DBA_GROUP'), $CFG->OLR_LOCATION);
   }
   s_set_perms ("0600", $CFG->OLR_LOCATION);

   #copy init.ohad,ohasd to init/rc dirs
   s_register_service('ohasd');
}

sub StartHA
{
   trace("Starting Oracle Restart");
   my $crsctl = crs_exec_path('crsctl');
   my $rc = system ("$crsctl start has");

    # Check if the service/daemon has started
    trace ("Checking ohasd");
    my $ohasd_running = check_service ("ohasd", 24);

    if ($ohasd_running) {
      trace ("ohasd started successfully");
    } else {
      print_error(199);
      exit 1;
    }
}

######################################################################
#                       M A I N                                      #
######################################################################

=head2 CRSPatch

   Performs post config steps for in-place/out-of-place patching of
   Grid Infrastructure home

=head3 Parameters

   None

=head3 Returns

   None

=cut

sub CRSPatch
{
  # To support rootcrs.pl -patch -destcrshome <home>
  # backward compatibility fucntion
  $CFG->NONROLLING(TRUE);
  $CFG->NORESTART(FALSE);
  crsPostPatch();
}

sub installPatchedScripts
{
   my $SUCC_REBOOT = 0;

   if (! isCkptexist("ROOTCRS_POSTPATCH_LOCKDSTHOME"))
   {
     writeCkpt("ROOTCRS_POSTPATCH_LOCKDSTHOME", CKPTSTART);
     $CFG->wipCkptName("ROOTCRS_POSTPATCH_LOCKDSTHOME");
   }

   if (! isCkptSuccess("ROOTCRS_POSTPATCH_LOCKDSTHOME"))
   {
     $CFG->wipCkptName("ROOTCRS_POSTPATCH_LOCKDSTHOME");

     trace("Instantiating and installing patched files");
     # Instantiate the patched files.
     Instantiatepatchfiles();

     writeCkpt("ROOTCRS_POSTPATCH_LOCKDSTHOME", CKPTSUC);
     $CFG->wipCkptName("ROOTCRS_POSTPATCH");
   }

   if (! isCkptexist("ROOTCRS_POSTPATCH_ACFSINST"))
   {
     writeCkpt("ROOTCRS_POSTPATCH_ACFSINST", CKPTSTART);
     $CFG->wipCkptName("ROOTCRS_POSTPATCH_ACFSINST");
   }

   if (! isCkptSuccess("ROOTCRS_POSTPATCH_ACFSINST"))
   {
     $CFG->wipCkptName("ROOTCRS_POSTPATCH_ACFSINST");

     trace("Installing ACFS drivers ...");
     $SUCC_REBOOT = crspatch_install_usm("crs");
     if (1 == $SUCC_REBOOT)
     {
       set_bold();
       die(dieformat(400));
       reset_bold();
     }

     writeCkpt("ROOTCRS_POSTPATCH_ACFSINST", CKPTSUC);
     $CFG->wipCkptName("ROOTCRS_POSTPATCH");
   }

   #
   # NOTE: This is commented out for 12.1 Post 12.1
   # branching, it will be enabled.
   #
   #if (! isCkptexist("ROOTCRS_POSTPATCH_OKAINST"))
   #{
   #  writeCkpt("ROOTCRS_POSTPATCH_OKAINST", CKPTSTART);
   #  $CFG->wipCkptName("ROOTCRS_POSTPATCH_OKAINST");
   #}
   #
   #if (! isCkptSuccess("ROOTCRS_POSTPATCH_OKAINST"))
   #{
   #  $CFG->wipCkptName("ROOTCRS_POSTPATCH_OKAINST");
   #
   #  trace("Installing OKA drivers ...");
   #
   #  # NOTE: Only for reboot failures, we will
   #  # exit. For rest of the failures, we
   #  # continue. Later, during resource OKA start,
   #  # we could fail and there too, we continue.
   #  #
   #  $SUCC_REBOOT = crspatch_install_oka();
   #  if (1 == $SUCC_REBOOT)
   #  {
   #    set_bold();
   #    die(dieformat(400));
   #    reset_bold();
   #  }
   #
   #  writeCkpt("ROOTCRS_POSTPATCH_OKAINST", CKPTSUC);
   #  $CFG->wipCkptName("ROOTCRS_POSTPATCH");
   #}

   if (! isCkptexist("ROOTCRS_POSTPATCH_AFDINST"))
   {
     writeCkpt("ROOTCRS_POSTPATCH_AFDINST", CKPTSTART);
     $CFG->wipCkptName("ROOTCRS_POSTPATCH_AFDINST");
   }

   if (! isCkptSuccess("ROOTCRS_POSTPATCH_AFDINST"))
   {
     $CFG->wipCkptName("ROOTCRS_POSTPATCH_AFDINST");

     trace("Installing AFD drivers ...");
     $SUCC_REBOOT = crspatch_install_afd();
     if (REBOOT == $SUCC_REBOOT)
     {
       set_bold();
       die(dieformat(400));
       reset_bold();
     }

     writeCkpt("ROOTCRS_POSTPATCH_AFDINST", CKPTSUC);
     $CFG->wipCkptName("ROOTCRS_POSTPATCH");
   }
}


=head2 perform_oop_steps 

   Performs post config steps for out of place patching of Grid Infrastructure home

=head3 Parameters

   [0] CRS home to be patched 

=head3 Returns

   None

=cut

sub perform_oop_steps
{
   my $patchGIHome = $_[0];

   # Perform CHM patching
   if (!perform_CHM_upgrade($patchGIHome)) {
     trace("Failed to patch Cluster Health Monitor.");
     print_error(408);
   }

   # update olr.loc
   s_validate_olrconfig($CFG->OLR_LOCATION, $patchGIHome);
}

=head2 HAPatch

   Performs post config steps for in place patching of  Oracle Restart home

=head3 Parameters

   None

=head3 Returns

   None

=cut

sub HAPatch
{

   trace ("Patching Oracle Restart");

   #Instantiate the patched files.
   Instantiatepatchfiles();

   my $SUCC_REBOOT = crspatch_install_usm("has"); 
   my $SUCC_REBOOT_AFD = crspatch_install_afd(); 

   clscfgLocalPatch($CFG->ORA_CRS_HOME)
     || die(dieformat(180, 'clscfg -localpatch', '-1'));

   if (! $CFG->NORESTART)
   {
     StartHA();
   }

   if (1 == $SUCC_REBOOT)
   {
     print color 'bold';
     print_error(400);
     print color 'reset';
   }
   if (REBOOT == $SUCC_REBOOT_AFD)
   {
     print color 'bold';
     print_error(400);
     print color 'reset';
   }
}

=head2 crspatch_install_usm 

   Install USM drivers

=head3 Parameters

   $1 : has/crs - this signifies the patch is patching Oracle Restart or RAC.

=head3 Returns

   1: Need to reboot after installation
   0: No need to reboot after installation

=cut

sub crspatch_install_usm
{
  my $SUCC_REBOOT = 0;
  my $has = $_[0];

  trace("Installing ACFS drivers to system root");
  my $ret = installUSMDriver($has);
  if (FAILED == $ret)
  {
    die(dieformat(196));
  }
  elsif (REBOOT == $ret)
  {
    trace("ACFS drivers cannot be installed, and reboot may resolve this");
    $SUCC_REBOOT = 1; 
  }
  else
  {
    trace("ACFS drivers installation completed");
  }

  return $SUCC_REBOOT;
}

=head2 crspatch_install_oka

   Install OKA drivers

=head3 Parameters

=head3 Returns

   1: Need to reboot after installation
   0: No need to reboot after installation

=cut

sub crspatch_install_oka
{
  my $SUCC_REBOOT = 0;
  my $has = "crs";

  if (isOKASupported())
  {
    trace("Installing OKA drivers to system root");

    my $ret = installOKADriver($has);
    if (REBOOT == $ret)
    {
      trace("OKA drivers cannot be installed, and reboot may resolve this");
      $SUCC_REBOOT = 1;
    }
    elsif (SUCCESS == $ret)
    {
      trace("OKA drivers installation completed");
    }
    else
    {
      trace("OKA drivers cannot be installed, proceeding with rest of the patch action");
    }
  }
  else 
  {
    trace("OKA is not supported");
  }

  return $SUCC_REBOOT;
}

=head2 crspatch_install_afd 

   Install AFD drivers

=head3 Returns

   REBOOT(3): Need to reboot after installation
   0: No need to reboot after installation

=cut

sub crspatch_install_afd
{
  my $SUCC_REBOOT = 0;
   #TODO: shmubeen remove later - ORA_ENABLE_AFD_INSTALL check
   my $afdInstall = FALSE;

   $afdInstall = uc($ENV{'ORA_ENABLE_AFD_INSTALL'});
   if ( $afdInstall ne "TRUE" )
   {
     trace("AFD disabled because of ENV in test mode");
     return SUCCESS;
   }

  if (isAFDSupported())
  {
    trace("Installing ASM Filter drivers to system root");
    my $ret = installAFDriver();
    if (FAILED == $ret)
    {
      print_error(445);
    }
    else
    {
      trace("ASM Filter drivers installation completed");
      if (REBOOT == $ret)
      {
        trace("Set reboot flag to 1");
        $SUCC_REBOOT = REBOOT;
      }
    }
  }
  else {
    trace("AFD is not supported");
  }

  return $SUCC_REBOOT;
}


sub unlockHAHomeforpatch
{
  my $exclfile = catfile($CFG->ORA_CRS_HOME, 'OPatch', 'crs', 'installPatch.excl');
  unlockHAHomefordeinstall($exclfile);
}

sub unlockInPlaceGIHome
{
  my $unlock_home = $_[0];

  trace("Unlocking the GI Home (inplace) for patching: $unlock_home");

  unlockHome($CFG->params('ORACLE_OWNER'),
              $CFG->params('ORA_DBA_GROUP'), 755, $unlock_home);
  print_error(347, $unlock_home);
}

sub unlockPatchDestHome
{
   my $unlock_home = $_[0];

   trace("Unlcoking the GI home: $unlock_home");
 
   modifyparamfile("ORACLE_HOME", $unlock_home);
   modifyparamfile("CRFHOME", $unlock_home);
   modifyparamfile("JREDIR", "$unlock_home/jdk/jre/");
   modifyparamfile("JLIBDIR", "$unlock_home/jlib");

   # set new home instantiate prior to unlock
   $ENV{'ORACLE_HOME'} = $unlock_home;
   $CFG->ORA_CRS_HOME($unlock_home);

   # instantiate prior to unlock
   instantiate_scripts($unlock_home);
  
   unlockHome($CFG->params('ORACLE_OWNER'),
              $CFG->params('ORA_DBA_GROUP'), 755, $unlock_home);
   print_error(347, $unlock_home);

}

#################################################################
## Function to integrate pre-patch steps for GI Home
## -prepatch [-nonrolling] [-norestart] [-destcrshome]
#################################################################
sub crsPrePatch
{
  trace("Performing the pre-patching steps required for GI stack ...");

  my $configuredGIHome = s_get_olr_file("crs_home");
  my $patchGIHome = ($CFG->DESTCRSHOME) ? ($CFG->DESTCRSHOME) : $configuredGIHome;

  my $dstcrshome = $CFG->DESTCRSHOME;
  trace("Destination CRS home: $dstcrshome");
  trace("Configured GI home: $configuredGIHome");
  trace("GI Home to be patched: $patchGIHome");

  if (($CFG->DESTCRSHOME) && ($patchGIHome eq $configuredGIHome))
  {
    trace("The destination GI home should not be the same as the configured "
         ."GI home for out-of-place GI home patching");
    die(dieformat(432));
  }

  if (! $CFG->NONROLLING) { prechecksForPatching($configuredGIHome); }

  crsPrePatchCkpts();

  ## if out-of-place patching
  if ($CFG->DESTCRSHOME)
  {
     trace("Preparing the destination GI home for out-of-place patching ...");
     unlockPatchDestHome($patchGIHome);

     trace("Done - Performing pre-pathching steps required for GI stack");
     writeCkpt("ROOTCRS_PREPATCH", CKPTSUC);

     return SUCCESS;
  }

  ## if in-place patching
  my $stackUp = checkGIStack($configuredGIHome);

  trace("Preparing the GI stack for in-place patching ...");
  $stackUp = performPrePatch($configuredGIHome, $stackUp);

  if (GI_STACK_DOWN != $stackUp)
  {
      trace("Stopping Oracle Clusterware stack ...");
      stopFullStack("force", $configuredGIHome) || die(dieformat(349));
  }

  ## Have an API to check if crsctl is use by EM GC. 
  ## Use fuser crsctl to find processes

  trace("Unlocking the GI home $patchGIHome for patching ...");
  unlockInPlaceGIHome($patchGIHome);

  trace("Done - Performing pre-pathching steps required for GI stack");
  writeCkpt("ROOTCRS_PREPATCH", CKPTSUC);

  return SUCCESS;
}

####################################################################
# This function peforms the following tasks:
# If rolling:
#   1) Starts the GI stack if required
#   2) Runs 'crsctl start rollingpatch'
# else if '-norestart' && stack is not up:
#   Starts OHASD only
#
# If '-norestart':
#   Sets CRSD to not start application resources
####################################################################
sub performPrePatch
{
  my $patchGIHome = $_[0];
  my $stackUp = $_[1];
  trace("Patch GI home: '$patchGIHome'");

  if (! $CFG->NONROLLING) 
  {
     if (GI_STACK_UP != $stackUp)
     {
       trace("Starting Oracle Clusterware stack to begin rolling patch...");
       startFullStack($patchGIHome) || die(dieformat(117));
       $stackUp = GI_STACK_UP;
     }      

     my $softwarepatch = getSoftwarePatch($patchGIHome);
     writeCkptProperty("ROOTCRS_PREPATCH", "SOFTWAREPATCH", $softwarepatch); 
     
     # Show cluster active patch level before rolling patch 
     my $CRSCTL = catfile($patchGIHome, 'bin', 'crsctl');
     system_cmd($CRSCTL, 'query', 'crs', 'activeversion', '-f');

     trace("Setting the GI stack for rolling patch application ...");
     startRollingPatch($patchGIHome) ||  die(dieformat(430));
  }
  elsif ($CFG->NORESTART) # non-rolling
  {
    if (GI_STACK_UP != $stackUp)
    {
      trace("OHASD needs to be up for disabling CRS resource");
      startOhasdOnly($patchGIHome) || die(dieformat(117));
      $stackUp = GI_STACK_UP;
    }
  }

  if ($CFG->NORESTART)
  {
     trace("Set the attribute to start CRS without starting the resources");
     setCRSResourceUseAttr($patchGIHome, 'disable') ||
       die(dieformat(180, 'crsctl set resource use', '-1'));
  }

  return $stackUp;
}

####################################################################
# This function peforms the following tasks:
#   1) Call perfromPrePatch if OOP patching
#   2) Stop old stack if needed before switching to the new stack
#   3) Lock destination GI home
#   4) Update software patch level in OLR using 'clscfg -localpatch'
#   5) Start the stack from destination GI home 
#   6) Update software patch level in OCR using 'clscfg -patch'
#   7) Run 'crsctl stop rollingpatch'
#   8) actionACFSDriversResource
#   9) actionOKADriversResource
#  10) Enable CRS resources and stop the stack if 'norestart' is specified
####################################################################
sub performPostPatch
{
  my $configuredGIHome = $_[0];
  my $patchGIHome = $_[1];

  trace("Configured GI home: '$configuredGIHome'");
  trace("patch GI home: '$patchGIHome'");

  my $stackUp = checkGIStack($configuredGIHome);

  ## if out-of-place & rolling patching
  if ($CFG->DESTCRSHOME) 
  {
    if (! isCkptexist("ROOTCRS_POSTPATCH_OOP_PRESTEPS"))
    {
       writeCkpt("ROOTCRS_POSTPATCH_OOP_PRESTEPS", CKPTSTART);
       $CFG->wipCkptName("ROOTCRS_POSTPATCH_OOP_PRESTEPS");
    }

    if (! isCkptSuccess("ROOTCRS_POSTPATCH_OOP_PRESTEPS"))
    {
      $CFG->wipCkptName("ROOTCRS_POSTPATCH_OOP_PRESTEPS");

      trace("Perform steps required for preparing GI stack for out-of-place patching");
      $stackUp = performPrePatch($configuredGIHome, $stackUp);

      writeCkpt("ROOTCRS_POSTPATCH_OOP_PRESTEPS", CKPTSUC);
      $CFG->wipCkptName("ROOTCRS_POSTPATCH");
    }
  }

  ## Must stop old stack 
  if (GI_STACK_DOWN != $stackUp)
  {
    trace("Stopping Oracle Clusterware stack ...");
    stopFullStack("force", $configuredGIHome) || die(dieformat(349));
  }

  if ($CFG->DESTCRSHOME)
  {
    if (! isCkptexist("ROOTCRS_POSTPATCH_OOP_REQSTEPS"))
    {
       writeCkpt("ROOTCRS_POSTPATCH_OOP_REQSTEPS", CKPTSTART);
       $CFG->wipCkptName("ROOTCRS_POSTPATCH_OOP_REQSTEPS");
    }

    if (! isCkptSuccess("ROOTCRS_POSTPATCH_OOP_REQSTEPS"))
    {
      $CFG->wipCkptName("ROOTCRS_POSTPATCH_OOP_REQSTEPS");

      trace("Performing steps required for out-of-place patching ...");
      perform_oop_steps($patchGIHome); 

      writeCkpt("ROOTCRS_POSTPATCH_OOP_REQSTEPS", CKPTSUC);
      $CFG->wipCkptName("ROOTCRS_POSTPATCH");
    }
  }
  
  installPatchedScripts();

  ## Call the following block for rolling and non-rolling cases
  clscfgLocalPatch($patchGIHome) || die(dieformat(180, 'clscfg -localpatch', '-1'));

  trace("Starting Oracle Clusterware stack for rolling patch ...");
  startFullStack($patchGIHome) || die(dieformat(117));

  clscfgPatch($patchGIHome) || die(dieformat(180, 'clscfg -patch', '-1'));
  stopRollingPatch($patchGIHome) ||  die(dieformat(431));

  # Show cluster active patch level
  my $CRSCTL = catfile($patchGIHome, 'bin', 'crsctl');
  system_cmd($CRSCTL, 'query', 'crs', 'activeversion', '-f');

  actionACFSDriversResource("start") || print_error(426);

  #actionOKADriversResource("start") || print_error(2009);

  if ($CFG->NORESTART) 
  {
    trace("Re-enable RESOURCE_USE_ENABLED after the rolling patch is finised");
    setCRSResourceUseAttr($patchGIHome, 'enable') ||
      die(dieformat(180, 'crsctl set resource use', '-1'));

    trace("Stop Oracle Clusterare because 'norestart' is specified"); 
    stopFullStack("force", $patchGIHome) || die(dieformat(349));
  }

  return SUCCESS;
}

#################################################################
## Function to integrate post-patch steps for GI Home
## -postpatch [-nonrolling] [-norestart] [-destcrshome]
#################################################################
sub crsPostPatch
{
  trace("Performing the post patching steps....");

  my $configuredGIHome = s_get_olr_file("crs_home");
  my $patchGIHome = ($CFG->DESTCRSHOME) ? ($CFG->DESTCRSHOME) : $configuredGIHome;

  my $dstcrshome = $CFG->DESTCRSHOME;
  trace("Destination CRS home: $dstcrshome");
  trace("Configured GI home: $configuredGIHome");
  trace("GI Home to be patched: $patchGIHome");

  if (($CFG->DESTCRSHOME) && ($patchGIHome eq $configuredGIHome))
  {
    if (isCkptexist("ROOTCRS_POSTPATCH_OOP_REQSTEPS") &&
         isCkptSuccess("ROOTCRS_POSTPATCH_OOP_REQSTEPS"))
    {
      trace("It has already switched to GI home being patched");
    }
    else
    {
      trace("The destination GI home should not be the same as the configured "
           ."GI home for out-of-place GI home patching");
      die(dieformat(432));   
    }
  }

  if (! $CFG->CRSPatch) { crsPostPatchCkpts($configuredGIHome); }
  performPostPatch($configuredGIHome, $patchGIHome);

  writeCkpt("ROOTCRS_POSTPATCH", CKPTSUC);
  return SUCCESS;
}

sub startRollingPatch
{
  my $crshome = shift;
  my $suc = FALSE;
  my $CRSCTL = catfile($crshome, 'bin', 'crsctl');

  trace("Setting the cluster in rolling patch mode ...");
  my @output = system_cmd_capture($CRSCTL, 'start', 'rollingpatch');
  my $rc = shift @output; 
  
  if (0 == $rc)
  {
    $suc = TRUE;
    trace("Successfully set the cluster in rolling patch mode");
  }
  else
  {
    if (scalar(grep(/1152/, @output)) > 0)
    {
      $suc =TRUE;
      trace("The cluster is already in rolling patch mode");
    }
  }

  if(!$suc)
  {
    print_lines(@output);
  }
  
  return $suc;
}

sub stopRollingPatch
{
  my $crshome = shift;
  my $suc = FALSE;
  my $CRSCTL = catfile($crshome, 'bin', 'crsctl');

  trace("Transition the cluster out of rolling patch mode");
  my @output = system_cmd_capture($CRSCTL, 'stop', 'rollingpatch');
  my $rc = shift @output;

  if (0 == $rc)
  {
    $suc = TRUE;
    trace("Successfully transition the cluster out of rolling patch mode");
  }
  else
  {
    if (scalar(grep(/1162/, @output)) > 0)
    {
      $suc = TRUE;
      trace("The patch has not been applied on all nodes in the cluster");
    }
    elsif (scalar(grep(/1171/, @output)) > 0)
    {
      $suc = TRUE;
      trace("Distinct patches have been applied on some of nodes");
    }
    elsif (scalar(grep(/1169/, @output)) > 0)
    {
      $suc = TRUE;
      trace("The cluster is consistent and there is no need to "
           ."transition the cluster out of rolling patch mode");
    }
  }

  if(!$suc)
  {
    print_lines(@output);
  }

  return $suc;
}

sub clscfgLocalPatch
{
  my $crshome = shift;
  my $suc = FALSE;
  my $CLSCFG = catfile($crshome, 'bin', 'clscfg');

  trace("Patch an existing configuration in OLR");
  my @output = system_cmd_capture($CLSCFG, '-localpatch');
  my $rc = shift @output;
  if (0 == $rc)
  {
    $suc = TRUE;
    trace("Successfully patched an existing configuration in OLR");
  }

  return $suc;
}

sub clscfgPatch
{
  my $crshome = shift;
  my $suc = FALSE;
  my $CLSCFG = catfile($crshome, 'bin', 'clscfg');

  trace("Patch an existing configuration");
  my @output = system_cmd_capture($CLSCFG, '-patch');
  my $rc = shift @output;
  if (0 == $rc)
  {
    $suc = TRUE;
    trace("Successfully patched an existing configuration");
  }

  return $suc;
}

sub setCRSResourceUseAttr 
{
  my $crshome = $_[0];
  my $action  = $_[1];

  my $suc = FALSE;
  my $CRSCTL = catfile($crshome, 'bin', 'crsctl');
  my $enable = 0;
 
  if ('enable' eq $action) {
    $enable = 1;
  }
  elsif ('disable' eq $action) {
    $enable = 0;
  }
  else {
    trace("Invalid action passed in: $action");
    return $suc;
  } 

  trace("Set RESOURCE_USE_ENABLED to $enable");
  my @output = system_cmd_capture($CRSCTL, 'set', 'resource', 'use', $enable);
  my $rc = shift @output;
  if (0 == $rc)
  {
    $suc = TRUE;
    trace("Successfully set RESOURCE_USE_ENABLED to $enable");
  }

  return $suc;
}

sub crsPrePatchCkpts
{
  my $destcrshome = ($CFG->DESTCRSHOME) ? ($CFG->DESTCRSHOME) : "null";
  my $nonrolling = ($CFG->NONROLLING) ? ($CFG->NONROLLING) : 0;
  my $norestart = ($CFG->NORESTART) ? ($CFG->NORESTART) : 0;
  
  writeCkpt("ROOTCRS_PREPATCH", CKPTSTART);
  writeCkptProperty("ROOTCRS_PREPATCH", "NONROLLING", $nonrolling);
  writeCkptProperty("ROOTCRS_PREPATCH", "DESTCRSHOME", $destcrshome);
  writeCkptProperty("ROOTCRS_PREPATCH", "NORESTART", $norestart); 
  $CFG->wipCkptName("ROOTCRS_PREPATCH");
}

sub crsPostPatchCkpts
{
  my $patchGIHome = $_[0];

  my $destcrshome = ($CFG->DESTCRSHOME) ? ($CFG->DESTCRSHOME) : "null";
  my $nonrolling = ($CFG->NONROLLING) ? ($CFG->NONROLLING) : 0;
  my $norestart = ($CFG->NORESTART) ? ($CFG->NORESTART) : 0;
  my $softwarepatch;

  if (! is_dev_env())
  {
    if ((! isCkptexist("ROOTCRS_PREPATCH")) ||
         (CKPTSUC ne getCkptStatus("ROOTCRS_PREPATCH")))
    {
      die("The post patching cannot be run because the pre patching ".
          "didn't complete successfully\n");
    }

    my $destcrshome2 = getCkptPropertyValue("ROOTCRS_PREPATCH", "DESTCRSHOME");
    my $nonrolling2 = getCkptPropertyValue("ROOTCRS_PREPATCH", "NONROLLING");

    if (($destcrshome ne trim($destcrshome2)) ||
         ($nonrolling != trim($nonrolling2)))
    {
      die("Improper options were specified for command ".
          "'rootcrs.pl -postpatch'\n");
    }

    if (! $nonrolling)
    {
      $softwarepatch = getCkptPropertyValue("ROOTCRS_PREPATCH", "SOFTWAREPATCH");
      my $releasepatch = getReleasePatch($patchGIHome);
      trace("release patch level: $releasepatch");
      trace("software patch level: $softwarepatch");
      if ($releasepatch eq trim($softwarepatch))
      {
        die("Oracle Clusterware patch level is same as release patch level ".
            "on the local node\n");
      }
    }
  }

  writeCkpt("ROOTCRS_POSTPATCH", CKPTSTART);
  writeCkptProperty("ROOTCRS_POSTPATCH", "NONROLLING", $nonrolling);
  writeCkptProperty("ROOTCRS_POSTPATCH", "DESTCRSHOME", $destcrshome);
  writeCkptProperty("ROOTCRS_POSTPATCH", "NORESTART", $norestart);

  my $releasepatch = getReleasePatch($patchGIHome);
  if (isCkptPropertyExists("ROOTCRS_POSTPATCH", "RELEASEPATCH"))
  {
    my $ckptrelpatch = getCkptPropertyValue("ROOTCRS_POSTPATCH", "RELEASEPATCH");
    trace("ckptrelpatch: [$ckptrelpatch]; releasepatch: [$releasepatch]");
    if ($ckptrelpatch != $releasepatch)
    {
      trace("Reset checkpoints for separate patch rounds");
      if (isCkptexist("ROOTCRS_POSTPATCH_LOCKDSTHOME"))
      {
        writeCkpt("ROOTCRS_POSTPATCH_LOCKDSTHOME", CKPTSTART);
      }
      if (isCkptexist("ROOTCRS_POSTPATCH_ACFSINST"))
      {
        writeCkpt("ROOTCRS_POSTPATCH_ACFSINST", CKPTSTART);
      }
      #if (isCkptexist("ROOTCRS_POSTPATCH_OKAINST"))
      #{
      #  writeCkpt("ROOTCRS_POSTPATCH_OKAINST", CKPTSTART);
      #}
      if (isCkptexist("ROOTCRS_POSTPATCH_AFDINST"))
      {
        writeCkpt("ROOTCRS_POSTPATCH_AFDINST", CKPTSTART);
      }
      if (isCkptexist("ROOTCRS_POSTPATCH_OOP_PRESTEPS"))
      {
        writeCkpt("ROOTCRS_POSTPATCH_OOP_PRESTEPS", CKPTSTART);
      }
      if (isCkptexist("ROOTCRS_POSTPATCH_OOP_REQSTEPS"))
      {
        writeCkpt("ROOTCRS_POSTPATCH_OOP_REQSTEPS", CKPTSTART);
      }
      writeCkptProperty("ROOTCRS_POSTPATCH", "RELEASEPATCH", $releasepatch);
    }
  }
  else
  {
    trace("Write current releasepatch level [$releasepatch] into checkpoints");
    writeCkptProperty("ROOTCRS_POSTPATCH", "RELEASEPATCH", $releasepatch);
  }

  $CFG->wipCkptName("ROOTCRS_POSTPATCH");
}

sub getSoftwarePatch
{
  my $crshome = $_[0];
  my $node = $_[1];
  my $sftpatch;

  trace("setting ORAASM_UPGRADE to 1");
  $ENV{'ORAASM_UPGRADE'} = "1";

  my $nodename = ($node) ? $node: "";
  my $CRSCTL = catfile($crshome, 'bin', 'crsctl');
  my @output = system_cmd_capture($CRSCTL, 'query', 'crs',
                                    'softwarepatch', $nodename);
  my $rc = shift @output;
  if (0 == $rc)
  {
    if ($output[0] =~ / \[(.+)\]/)
    {
      $sftpatch = $1;
    }
  }

  trace("Oracle Clusterware patch level on node '$nodename' is [$sftpatch]");
  return $sftpatch;
}

sub getReleasePatch
{
  my $crshome = $_[0];
  my $relpatch;

  my $CRSCTL = catfile($crshome, 'bin', 'crsctl');
  my @output = system_cmd_capture($CRSCTL, 'query', 'crs', 'releasepatch');
  my $rc = shift @output;
  if (0 == $rc)
  {
    if ($output[0] =~ / \[(.+)\].+\[(.*)\]/)
    {
      $relpatch = $1;
    }
    elsif ($output[0] =~ / \[(.+)\]/)
    {
      $relpatch = $1;
    }
  }

  trace("Oracle Clusterware release patch level is [$relpatch]");
  return $relpatch;
}

sub getActivePatchLevel
{
  my $crshome = $_[0];
  my $apl;

  trace("setting ORAASM_UPGRADE to 1");
  $ENV{'ORAASM_UPGRADE'} = "1";

  my $CRSCTL = catfile($crshome, 'bin', 'crsctl');
  my @output = system_cmd_capture($CRSCTL, 'query', 'crs', 'activeversion', '-f');
  my $rc = shift @output;
  if (0 == $rc)
  {
    if ($output[0] =~ /\[(.+)\].+\[(.+)\].+\[(.+)\]/)
    {
      $apl = $3;
    }
  }

  trace("Oracle Clusterware active patch level is [$apl]");
  return $apl;
}

# This subroutine only works during prepatch.
sub isFirstNodeToPatch
{
  my $crshome = $_[0];
  my $activePatchLevel = getActivePatchLevel($crshome);

  my @clunodes = split(',', $CFG->params('NODE_NAME_LIST'));
  foreach my $node (@clunodes)
  {
     my $iNodePatchLevel = getSoftwarePatch($crshome, $node);
    if ($activePatchLevel != $iNodePatchLevel)
    {
      trace("Node '$node' has been patched to [$iNodePatchLevel]");
      return FALSE;
    }
  }
  
  trace("First node to patch");
  return TRUE;
}

# This subroutine only works during prepatch.
# Note: this subroutine will return FALSE if used during postpatch
# on the last node and after APL has been updated. DO NOT use in postpatch.
sub isLastNodeToPatch
{
  my $crshome = $_[0];
  my $host = tolower_host();
  my $localNodePatchLevel = getSoftwarePatch($crshome, $host);
  
  my $rmtNodePatchLevel = undef;
  my @clunodes = split(',', $CFG->params('NODE_NAME_LIST'));
  foreach my $node (@clunodes)
  {
    if (lc($node) =~ /\b$host\b/i) { next; }

    my $iNodePatchLevel = getSoftwarePatch($crshome, $node);
    if ($iNodePatchLevel == $localNodePatchLevel)
    {
      trace("The local node has the same software patch".
            " level [$localNodePatchLevel] as remote node '$node'"); 
      return FALSE;
    }

    if ((defined($rmtNodePatchLevel)) &&
          ($iNodePatchLevel != $rmtNodePatchLevel))
    {
      trace("The remote node '$node' has different software patch".
            " level [$iNodePatchLevel] than other remote nodes");
      return FALSE;
    }

    $rmtNodePatchLevel = $iNodePatchLevel;
  }

  trace("Last node to patch");
  return TRUE;
}

sub prechecksForPatching
{
  my $crshome = $_[0];
  
  if (isFirstNodeToPatch($crshome))
  {
    if (isBigCluster() && (! isHubNode()))
    {
      die(dieformat(455));
    }
  }
}



1;
